<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Balu
 * @copyright Copyright (c) 2015 - 2019 Baluart.COM
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link http://easyforms.baluart.com/ Easy Forms
 */

namespace app\components;

use app\controllers\user\RecoveryController;
use app\controllers\user\RegistrationController;
use app\helpers\ArrayHelper;
use app\helpers\MailHelper;
use app\models\forms\RegistrationForm;
use app\models\User;
use Da\User\Event\FormEvent;
use Da\User\Event\ResetPasswordEvent;
use Da\User\Event\UserEvent;
use Da\User\Module;
use Yii;
use yii\base\Application;
use yii\base\BootstrapInterface;
use yii\base\Event;

/**
 * Class Bootstrap
 * @package app\components
 */
class Bootstrap implements BootstrapInterface
{
    /**
     * Bootstrap method to be called during application bootstrap stage.
     * @param Application $app the application currently running
     */
    public function bootstrap($app)
    {

        $app->on(Application::EVENT_BEFORE_REQUEST, function () use ($app) {

            try {

                /*******************************
                /* Default Route
                /*******************************/
                if (isset(Yii::$app->user->isGuest) && Yii::$app->user->isGuest) {
                    $app->defaultRoute = 'user/security/login';
                }

                /*******************************
                /* Mailer
                /*******************************/

                // Change transport class to Sendinblue or SMTP
                $defaultMailerTransport = !empty($app->params['App.Mailer.transport']) ? $app->params['App.Mailer.transport'] : '';

                if ($app->settings->get('mailerTransport', 'app', $defaultMailerTransport) === 'sendinblue') {

                    // Set Sendinblue mail component as mailer
                    $app->set('mailer', [
                        'class' => 'app\components\sendinblue\Mailer',
                        'apiKey' => $app->settings->get('sendinblue.key'),
                    ]);

                } else {

                    // Set default transport class with PHP mail()
                    $transport = [
                        'class' => 'Swift_MailTransport',
                    ];

                    // Set an SMTP account
                    if ($app->settings->get('mailerTransport', 'app', $defaultMailerTransport) === 'smtp') {
                        $transport = [
                            'class' => 'Swift_SmtpTransport',
                            'host' => $app->settings->get("smtp.host"),
                            'username' => $app->settings->get("smtp.username"),
                            'password' => base64_decode($app->settings->get("smtp.password")),
                            'port' => $app->settings->get("smtp.port"),
                            'encryption' => $app->settings->get("smtp.encryption") == 'none'?
                                null :
                                $app->settings->get("smtp.encryption"),
                        ];
                    }

                    // Set mail queue component as mailer
                    $app->set('mailer', [
                        'class' => 'app\components\queue\MailQueue',
                        'mailsPerRound' => 10,
                        'maxAttempts' => 3,
                        'transport' => $transport,
                        'messageConfig' => [
                            'charset' => 'UTF-8',
                        ]
                    ]);
                }

                /*******************************
                /* User session
                /*******************************/

                if (isset($app->user) && !$app->user->isGuest) {
                    /** @var \app\models\Profile $profile */
                    $profile = $app->user->identity->profile;

                    // Setting the timezone to the current users timezone
                    if (isset($profile->timezone)) {
                        $app->setTimeZone($profile->timezone);
                    }

                    // Setting the language to the current users language
                    if (isset($profile->language)) {
                        $app->language = $profile->language;
                    }
                }

                /**
                 * User module
                 */
                /** @var Module $userModule */
                $userModule = $app->getModule('user');

                // Mail params
                if (isset($userModule->mailParams)) {
                    $userModule->mailParams = [
                        'fromEmail' => MailHelper::from(Yii::$app->settings->get("app.supportEmail")),
                        'welcomeMailSubject' => Yii::t('usuario', 'Welcome to {0}', Yii::$app->settings->get("app.name")),
                        'confirmationMailSubject' => Yii::t('usuario', 'Confirm account on {0}', Yii::$app->settings->get("app.name")),
                        'reconfirmationMailSubject' => Yii::t('usuario', 'Confirm email change on {0}', Yii::$app->settings->get("app.name")),
                        'recoveryMailSubject' => Yii::t('usuario', 'Complete password reset on {0}', Yii::$app->settings->get("app.name")),
                    ];
                }

                // Enable Registration
                if ($app->settings->get('anyoneCanRegister', 'app', false)) {
                    $userModule->enableRegistration = true;
                }

                // Enable Unconfirmed Email Login
                if ($app->settings->get('unconfirmedEmailLogin', 'app', true)) {
                    $userModule->allowUnconfirmedEmailLogin = true;
                }

                // Enable Two Factor Authentication
                if ($app->settings->get('twoFactorAuthentication', 'app', false)) {
                    $userModule->enableTwoFactorAuthentication = true;
                }

                // Max Password Age
                if ($maxPasswordAge = (int) $app->settings->get('maxPasswordAge', 'app', null)) {
                    if ($maxPasswordAge > 0) {
                        $userModule->maxPasswordAge = $maxPasswordAge;
                    }
                }

                /**
                 * Fix https issue with cloudflare when it's needed
                 */
                if (isset($_SERVER['HTTP_CF_VISITOR'])) {
                    if (preg_match('/https/i', $_SERVER['HTTP_CF_VISITOR'])) {
                        $_SERVER['HTTPS'] = 'On';
                        $_SERVER['HTTP_X_FORWARDED_PORT'] = 443;
                        $_SERVER['SERVER_PORT'] = 443;
                    }
                }

            } catch (\Exception $e) {
                // Do nothing
            }

        });

        /*******************************
        /* Event Handlers
        /*******************************/

        $app->on(
            'app.form.updated',
            ['app\events\handlers\FormEventHandler', 'onFormUpdated']
        );

        $app->on(
            'app.form.submission.received',
            ['app\events\handlers\SubmissionEventHandler', 'onSubmissionReceived']
        );

        $app->on(
            'app.form.submission.accepted',
            ['app\events\handlers\SubmissionEventHandler', 'onSubmissionAccepted']
        );

        $app->on(
            'app.form.submission.rejected',
            ['app\events\handlers\SubmissionEventHandler', 'onSubmissionRejected']
        );

        $app->on(
            'app.form.submission.verified',
            ['app\events\handlers\SubmissionEventHandler', 'onSubmissionVerified']
        );

        // After new user registration
        Event::on(RegistrationController::className(), FormEvent::EVENT_AFTER_REGISTER, function (FormEvent $event) {
            /** @var RegistrationForm $form */
            $form = $event->getForm();
            /** @var User $user */
            $user = User::find()->where(['email' => $form->email])->one();
            $profile = $user->profile;
            // Assign default user language to user
            if ($defaultUserLanguage = Yii::$app->settings->get('app.defaultUserLanguage')) {
                $profile->language = $defaultUserLanguage;
            }
            // Assign default user timezone to user
            if ($defaultUserTimezone = Yii::$app->settings->get('app.defaultUserTimezone')) {
                $profile->timezone = $defaultUserTimezone;
            }
            $profile->save(false);
            // Assign default user role to user
            if ($defaultUserRole = Yii::$app->settings->get('app.defaultUserRole')) {
                $auth = Yii::$app->authManager;
                $roles = $auth->getRoles();
                $roles = ArrayHelper::getColumn($roles, 'name');
                if (in_array($defaultUserRole, $roles)) {
                    $authRole = $auth->getRole($defaultUserRole);
                    $auth->assign($authRole, $user->id);
                }
            }
            // Redirect to login page
            Yii::$app->controller->redirect(['/user/security/login']);
            Yii::$app->end();
        });

        // After user confirmation
        Event::on(RegistrationController::className(), UserEvent::EVENT_AFTER_CONFIRMATION, function (UserEvent $event) {
            // Redirect to login page
            Yii::$app->controller->redirect(['/']);
            Yii::$app->end();
        });

        // After reset password
        Event::on(RecoveryController::class,FormEvent::EVENT_AFTER_REQUEST, function (FormEvent $event) {
            // Redirect to login page
            Yii::$app->controller->redirect(['/user/security/login']);
            Yii::$app->end();
        });
    }
}
